#!/usr/bin/env python
#
############################################################################
#
# MODULE:     r.sunangle.py
# AUTHOR(S):	Ivan Barka (ivan.barka (at) gmail.com, National Forest Centre Zvolen, Slovakia) in cooperation with Dusan Senko (Institute of Botany, Slovak Academy of Sciences)
# PURPOSE:    Computes sun angles for each point in given text file, from day of beginning to end day with given time increment
#             
#		This program is free software under the GNU General Public
#		License (>=v2). Read the file COPYING that comes with GRASS
#		for details.
#
# ----------------------------------------------------------------------------
# This Python script calculates solar incidence angle for a point location/s
# on an inclined or a horizontal plane
# for particular day and time/time increment.
# ----------------------------------------------------------------------------
# The algorithm employs the equations and terminology from:
# Hofierka, J., Suri, M. (2002, september 11). The solar radiation model for Open source GIS: implementation and applications. Trento, Italy.
# ----------------------------------------------------------------------------
# Inputs require a text file, which is read by this script 
# and the values contained enter the algorithm.
# The format of the input file should be:
# point;latitude;longitude;increment;day_start;day_end; separated with ;
# it is possible to append information on slope, aspect and elevation -
# if given and script is run with -s flag, then slope and aspect is taken from input
# otherwise it is taken from rasters specified by elev, slope and aspect parameters
# ----------------------------------------------------------------------------
# The results are saved in a new file_solar_incidence_angle.csv text file
# The format of the output file is:
# 1;point;latitude;longitude;elevation;slope_angle;aspect_angle;day;time;
# inclined_sun_angle;inclined_sunset_time;horiz_sun_angle;
# horiz_sunset_time;sun_azimuth;max_sun_angle;delta_sun_angle
# ----------------------------------------------------------------------------
# units - degrees, decimal hours, days from 1-365
# if aspect angle is to be considered the input values should follow
# traditional clockwise direction starting from the north=0, east=90 ...
# however, the article Hofierka and Suri (2002) says east is the beginning...
# to me the algorithm considers south to be 0 based on experiment
# ----------------------------------------------------------------------------
# the inclined_sunset_time can return NaN (no data) if (-C_33/C_31) 
# return value out of <-1;1>, e.g. arccos(-1.2) does not exist, 
# This problem needs a closer look:), but other variables are safe and reliable.
# ----------------------------------------------------------------------------
# if point is outside of the current region and slope and aspect are going to be read
# from rasters, 0 is used for slope and aspect; then * is writen to output
# in the fields elevation, slope_angle and aspect_angle
# ----------------------------------------------------------------------------
# Module is based on R script originally written by Michal Gallay (michal.gallay (at) gmail.com) for Institute of Botany Slovak Academy of Sciences in Bratislava, Slovakia.
# Script citation: Barka, I. Gallay, M., Senko, D. r.sunangle Python script in the GRASS GIS environment for calculation of solar radiation parameters. In: Merea P. et al., 2015: Ecological niche differentiation between tetra- and octoploids of Jacobaea vulgaris (Asteraceae). Preslia, Praha xx: xx-xx.
#-------------------------------------------------------------------------------
#############################################################################

#%module
#% description: calculates solar incidence angle for a point location(s) on an inclined or a horizontal plane for particular day and time/time increment.
#% keywords: raster
#% keywords: angle
#%end
#%option G_OPT_F_INPUT
#% key: input_file
#% type: string
#% description: Input txt file with points coordinates and time data
#% required: yes
#%end
#%option G_OPT_R_INPUT
#% key: elev
#% description: Name of elevation raster map
#% required : no
#%end
#%option G_OPT_R_INPUT
#% key: slope
#% description: Name of slope raster map
#% required : no
#%end
#%option G_OPT_R_INPUT
#% key: aspect
#% description: Name of aspect raster map
#% required : no
#%end
#%option G_OPT_F_OUTPUT
#% key: output_file
#% type: string
#% description: Output txt file for results
#% required: no
#%end
#%flag
#% key: s
#% description: Use terrain parameters from input file instead of raster maps
#%end


#import sys
import os
import grass.script as grass
import atexit
import string
from math import *
#import fnmatch

def cleanup():
    if tmp:
        grass.run_command('g.remove', rast = tmp, quiet = True)

def main():
    input_file = options['input_file']
    elevMap = options['elev'] # elevation map
    slopeMap = options['slope'] # raster with slope values
    aspectMap = options['aspect'] # raster with aspect values
    output_file = options['output_file']
    s = flags['s']
    
    if not input_file:
        grass.fatal(_("Required parameter <input_file> not set"))

    if not elevMap:
        elevMap = "dmr"
    
    if not slopeMap:
        slopeMap = "slope"

    if not aspectMap:
        aspectMap = "aspect"

    #check if input files exists
    if not os.path.isfile(input_file):
        grass.fatal(_("<%s> does not exist.") % input_file)
    if not grass.find_file(elevMap)['file'] and not s:
        grass.fatal(_("<%s> does not exist.") % elevMap)
    if not grass.find_file(slopeMap)['file'] and not s:
        grass.fatal(_("<%s> does not exist.") % slopeMap)
    if not grass.find_file(aspectMap)['file'] and not s:
        grass.fatal(_("<%s> does not exist.") % aspectMap)
    
    #mapset = grass.gisenv()['MAPSET']

    
    finput = open(input_file, "r")
    #creates a file with a header to save the results in
    if not output_file:
        foutput = open(string.replace(input_file, ".txt", "_solar_incidence_angle.csv"), "w")
    else:
        foutput = open(output_file, "w")
    foutput.write("point;latitude;longitude;elevation;slope_angle;aspect_angle;day;time;inclined_sun_angle;inclined_sunset_time;horiz_sun_angle;horiz_sunset_time;sun_azimuth;max_sun_angle;delta_sun_angle\n")  
    #--------- loop for file rows (particular point) ----------------
    linenum = 0
    for line in finput:
        linenum += 1
        try:
            # split values by ;
            lineValues = string.split(string.strip(line),";")
            
            #--------------- input variables -------------------
            point = lineValues[0]
            day_start = int(lineValues[4])      #which day from the 1. January
            day_end = int(lineValues[5])        #number of the day which terminates the period
            latitudeStr = lineValues[1]
            longitudeStr = float(lineValues[2])
            latit = radians(float(lineValues[1]))   #latitude in radians
            # transform latitude and longitude to etrs_laea
            coor = grass.read_command("m.proj", flags="id", coordinates="%s,%s" % (longitudeStr, latitudeStr))
            xcoorStr, ycoorStr = string.split(coor,"|")[0:2]
            #grass.message(_("Suradnice pre bod %s su %s,%s" % (point,xcoorStr,ycoorStr)))
            
            if s:
                elevationStr = lineValues[8]
                slope_angleStr = lineValues[6]
                aspect_angleStr = lineValues[7]
                slope_angle = radians(float(slope_angleStr))    #slope angle
                aspect_angle = float(aspect_angleStr)
                if aspect_angle <= 180.0:
                    aspect_angle = 180.0 - aspect_angle
                else:
                    aspect_angle = 360.0 - aspect_angle + 180.0
                aspect_angle = radians(aspect_angle)   #slope aspect angle in radians
            else:
                # get elevation value from dem raster
                out = grass.read_command("r.what", map=elevMap, coordinates="%s,%s" % (xcoorStr,ycoorStr))
                elevationStr = string.strip(string.split(out,"|")[3])
                # get slope value from slope raster
                out = grass.read_command("r.what", map=slopeMap, coordinates="%s,%s" % (xcoorStr,ycoorStr))
                slope_angleStr = string.strip(string.split(out,"|")[3])
                # get aspect angle from aspect raster
                out = grass.read_command("r.what", map=aspectMap, coordinates="%s,%s" % (xcoorStr,ycoorStr))
                aspect_angleStr = string.strip(string.split(out,"|")[3])
                try:
                    aspect_angle = float(aspect_angleStr)
                except:
                    aspect_angle = 0
                if aspect_angle <= 180.0:
                    aspect_angle = 180.0 - aspect_angle
                else:
                    aspect_angle = 360.0 - aspect_angle + 180.0
                try:
                    slope_angle = radians(float(slope_angleStr))
                except:
                    slope_angle = 0
                aspect_angle = radians(aspect_angle)   #slope aspect angle in radians
                
            local_time =     0          #time in decimal hours
            time_increment = float(string.split(line,";")[3])/60.0   #time increment in minutes converted to decimal hours
            #time_increment_dechour = float(string.split(line,";")[3])/60.0   #time increment in minutes converted to decimal hours
            period = day_end-day_start+2  #number of days
            #print point, day_start, day_end, latit, slope_angle, aspect_angle, period, time_increment_dechour
            #solar incidence angle and azimuth calculation
            #------------- loop for days -----------------
            for i in range(1,period):
                output = []
                angles = []
                # --------------- loop for time increment -----------
                
                for j in range(1, int(round((24.0/time_increment),0)) + 1):
                    day_angle = 2*pi*day_start/365.25
                    sun_declin = asin(0.3978*sin(day_angle - 1.4 + 0.0355*sin(day_angle - 0.0489)))
                    hour_angle = 0.261799*(local_time - 12)
                    rel_latit = asin((-cos(latit))*sin(slope_angle)*cos(aspect_angle)+sin(latit)*cos(slope_angle))
                    rel_longit = atan(-(sin(slope_angle)*sin(aspect_angle))/(sin(latit)*sin(slope_angle)*cos(aspect_angle)+cos(latit)*cos(slope_angle)))
    
                    C_31 = cos(rel_latit)*cos(sun_declin)
                    C_33 = sin(rel_latit)*sin(sun_declin)
                    C11 = sin(latit)*cos(sun_declin)
                    C13 = (-cos(latit))*sin(sun_declin)
                    C22 =  cos(sun_declin)
                    C31 = cos(latit)*cos(sun_declin)
                    C33 = sin(latit)*sin(sun_declin)
    
                    #calculates incidence angle for an inclined surface
                    inclined_sun_angle = asin(C_31*cos(hour_angle - rel_longit) + C_33)*180/pi
                    angles.append(inclined_sun_angle)
                    inclined_hour_angle = acos((-C_33)/C_31)+ rel_longit
                    inclined_sunset_time =  inclined_hour_angle/0.261799+12
    
                    #calculates incidence angle for a horizontal surface
                    horiz_sun_angle = asin(C31*cos(hour_angle)+C33)*180/pi
                    horiz_hour_angle = acos((-C33)/C31)
                    horiz_sunset_time = horiz_hour_angle/0.261799+12
    
                    #calculates solar azimuth, does not depend on the surface inclination
                    sun_azimuth = acos((C11*cos(hour_angle)+C13)/(((C22*sin(hour_angle))**2+(C11*cos(hour_angle)+C13)**2)**0.5))*180/pi
                    if local_time - (i-1)*24 < 12.0:
                        sun_azimuth = 180.0 - sun_azimuth
                    else:
                        sun_azimuth += 180.0
    
                    # append results to output
                    output.append("%s;%s;%s;%s;%s;%s;%d;%g;%.4f;%.4f;%.4f;%.4f;%.4f" % (point,latitudeStr,longitudeStr,elevationStr,slope_angleStr,aspect_angleStr,day_start,local_time - (i-1)*24,inclined_sun_angle,inclined_sunset_time,horiz_sun_angle,horiz_sunset_time,sun_azimuth))
    
                    # updates local_time and row pivot for the next loop
                    local_time += time_increment
                day_start += 1
                # compute max and deltas of inclined angles
                for i in range(0,len(angles)):
                    output[i] = "%s;%.4f;%.4f" % (output[i], max(angles), max(angles) - angles[i])
                # write day values to output file
                for o in output:
                    foutput.write("%s\n" % o)
        except:
            grass.message(_("Line number %d skipped, probably header or wrong formated data" % linenum))
    foutput.close()
    finput.close()

    grass.message(_("Done."))    

if __name__ == "__main__":
    options, flags = grass.parser()
    tmp = None
    atexit.register(cleanup)
    main()